// Test L-shaped embeddings of n-vertex trees in all point sets of size n.
// If the preprocessor variable ORDERED is set to 0, then the trees are
// considered unordered, whereas if ORDERED is set to 1, then the trees
// are considered ordered, with a particular fixed cyclic ordering of
// neighbors around each vertex.
// The trees are read from files, and the point sets are generated by
// the program.
// (c) 2018 Manfred Scheucher <scheucher@math.tu-berlin.de>


#include <algorithm>
#include <cassert>
#include <ctime>
#include <fstream>
#include <iomanip>
#include <iostream>
#include <sstream>
#include <thread>

#define DEBUG 0  // set to 1 or 2 for more verbose output


#define SGN(x) ((x > 0) - (x < 0))


int XPERM[N];


// data structure to represent embedding of tree on a point set
// (the point set is not part of this data structure)
typedef struct
{
    int mapping_V_to_X[N];  // x-coordinates of vertices
    int mapping_X_to_V[N];  // vertices at given x-coordinates
    char pixel[N][N];  // drawing of the tree on n times n grid
                       // '.'=empty cell, '|'=horizontal edge, '-'=vertical edge, '+'=edge turn, vertex labels)
#if ORDERED != 0
    int order[N][4];  // neighbors of each vertex in the four directions 0=right, 1=up, 2=left, 3=down
#endif
} drawing;


void print_time() 
{
    std::time_t t = std::time(nullptr);
    std::cout << "[" << std::put_time(std::gmtime(&t), "%F.%T") << "] ";
}


void print_drawing(drawing& D)
{
    for(int y=0;y<N;y++)
    {
        std::cout << "\t\t";
        for(int x=0;x<N;x++)
        {
            std::cout << D.pixel[x][y];
        }
        std::cout << "\n";
    }
    std::cout << "\n";
}


#if ORDERED != 0
    // Check if the drawing respects the prescribed ordering of vertices around v0.
    // This function uses the observation that for a degree-1 or degree-2 vertex,
    // all cyclic orderings are equivalent, and given a degree-3 or degree-4 vertex
    // whose neighbors are embedded in the ***wrong*** order, this can be detected
    // by looking at two outgoing edges that are neighbors in the cyclic list
    // (right, up, left, down).
    int valid_rotsys(int degree[N], int rotsys[][4], drawing& D, int v0, int v1, int j0)
    {
        int j0prev = (j0-1+degree[v0]) % degree[v0];
        int j0next = (j0+1) % degree[v0];
        int k0 = 0;
        for(;k0<degree[v0];k0++) if(rotsys[v0][k0] == v1) break;
        assert(rotsys[v0][k0] == v1);
        int k0prev = (k0-1+degree[v0]) % degree[v0];
        int k0next = (k0+1) % degree[v0];
        if(D.order[v0][j0prev] != -1 && D.order[v0][j0prev] != rotsys[v0][k0prev]) return 0;
        if(D.order[v0][j0next] != -1 && D.order[v0][j0next] != rotsys[v0][k0next]) return 0;
        return 1;
    }
#endif


// Recursively try embedding remaining tree on the given point set.
// Vertices 0,...,v1-1 have already been embedded before.
int test_inner(int Y[N], int parent[N], int degree[N], int rotsys[][4], drawing& D, int v1, int previous_mapping[N])
{
    if(v1 == N)
    {
        // successfully embedded all vertices
        for(int i=0;i<N;i++) previous_mapping[i] = D.mapping_V_to_X[i];
        if(DEBUG > 1) print_drawing(D);
        return 1;
    }
    else
    {
        // place vertex v1
        int v0 = parent[v1];  // parent of v1
        int x0 = D.mapping_V_to_X[v0];
        int y0 = Y[x0];
        
        // try all possible x-coordinates for vertex v1
        int x1ii=0;
        for(int x1i=0;x1i<N;x1i++)
        {
            int x1 = XPERM[x1i];

            if(previous_mapping[0] != -1)
            {
                if(x1i == 0)
                {
                    x1 = previous_mapping[v1];  // reuse x-coordinate from previous embedding
                }
                else 
                {
                    if(XPERM[x1ii] == previous_mapping[v1]) x1ii++;
                    x1 = XPERM[x1ii];
                    x1ii++;
                }
            }

            int y1 = Y[x1];
            if((degree[v1] == 4) && (x1 == 0 || x1 == N-1 || y1 == 0 || y1 == N-1)) continue;  // degree-4 vertices must not be embedded on the boundary

            if(D.mapping_X_to_V[x1] != -1) continue;  // skip already used x-coordinates
              
            int dx = SGN(x1-x0);
            int dy = SGN(y1-y0);
            assert(dx*dx == 1);
            assert(dy*dy == 1);
            {
                // try drawing horizontal connection first
                int valid = 1;
                if(valid)
                {
                    for(int x=x0+dx;x!=x1;x+=dx)
                    {
                        if(D.pixel[x][y0] != '.')
                        {
                            valid = 0;
                            break;
                        }
                    }
                }
                if(valid && D.pixel[x1][y0] != '.')
                {
                    valid = 0;
                }
                if(valid)
                {
                    // try continuing with vertical connection
                    for(int y=y0+dy;y!=y1;y+=dy)
                    {
                        if(D.pixel[x1][y] != '.')
                        {
                            valid = 0;
                            break;
                        }
                    }
                }
                if(valid) 
                {
                    // edge can be drawn this way, now try embedding the remaining vertices
                    D.mapping_V_to_X[v1] = x1;
                    D.mapping_X_to_V[x1] = v1;
                    for(int x=x0+dx;x!=x1;x+=dx) assert(D.pixel[x][y0] == '.');
                    for(int x=x0+dx;x!=x1;x+=dx) D.pixel[x][y0] = '-';
                    assert(D.pixel[x1][y0] == '.');
                    D.pixel[x1][y0] = '+';
                    for(int y=y0+dy;y!=y1;y+=dy) assert(D.pixel[x1][y] == '.');
                    for(int y=y0+dy;y!=y1;y+=dy) D.pixel[x1][y] = '|';
                    assert(D.pixel[x1][y1] == '.');
                    D.pixel[x1][y1] = '0'+v1;

                    #if ORDERED != 0
                        assert(D.order[v0][(dx>0?0:2)] == -1);
                        D.order[v0][(dx>0?0:2)] = v1;
                        assert(D.order[v1][(dy>0?1:3)] == -1);
                        D.order[v1][(dy>0?1:3)] = v0;

                        // check order of neighbors
                        int j0 = (dx>0?0:2);
                        if(!valid_rotsys(degree,rotsys,D,v0,v1,j0)) valid = 0;
                    #endif

                    if(valid)
                    {
                        if(test_inner(Y,parent,degree,rotsys,D,v1+1,previous_mapping)) return 1;  // embed remaining tree
                    }

                    D.mapping_V_to_X[v1] = -1;
                    D.mapping_X_to_V[x1] = -1;
                    for(int x=x0+dx;x!=x1;x+=dx) D.pixel[x][y0] = '.';
                    D.pixel[x1][y0] = '.';
                    for(int y=y0+dy;y!=y1;y+=dy) D.pixel[x1][y] = '.';
                    D.pixel[x1][y1] = '.';

                    #if ORDERED != 0
                        D.order[v0][(dx>0?0:2)] = -1;
                        D.order[v1][(dy>0?1:3)] = -1;
                    #endif
                }
            }
            {
                // try drawing vertical connection first
                int valid = 1;
                if(valid)
                {
                    for(int y=y0+dy;y!=y1;y+=dy)
                    {
                        if(D.pixel[x0][y] != '.')
                        {
                            valid = 0;
                            break;
                        }
                    }
                }
                if(valid && D.pixel[x0][y1] != '.')
                {
                    valid = 0;
                }
                if(valid)
                {
                    // try continuing with horizontal connection
                    for(int x=x0+dx;x!=x1;x+=dx)
                    {
                        if(D.pixel[x][y1] != '.')
                        {
                            valid = 0;
                            break;
                        }
                    }
                }
                if(valid) 
                {
                    // edge can be drawn this way, now try embedding the remaining vertices
                    D.mapping_V_to_X[v1] = x1;
                    D.mapping_X_to_V[x1] = v1;
                    for(int y=y0+dy;y!=y1;y+=dy) assert(D.pixel[x0][y] == '.');
                    for(int y=y0+dy;y!=y1;y+=dy) D.pixel[x0][y] = '|';
                    assert(D.pixel[x0][y1] == '.');
                    D.pixel[x0][y1] = '+';
                    for(int x=x0+dx;x!=x1;x+=dx) assert(D.pixel[x][y1] == '.');
                    for(int x=x0+dx;x!=x1;x+=dx) D.pixel[x][y1] = '-';
                    assert(D.pixel[x1][y1] == '.');
                    D.pixel[x1][y1] = '0'+v1;

                    #if ORDERED != 0
                        assert(D.order[v0][(dy<0?1:3)] == -1);
                        D.order[v0][(dy<0?1:3)] = v1;
                        assert(D.order[v1][(dx<0?0:2)] == -1);
                        D.order[v1][(dx<0?0:2)] = v0;

                        // check order of neighbors
                        int j0 = (dy<0?1:3);
                        if(!valid_rotsys(degree,rotsys,D,v0,v1,j0)) valid = 0;
                    #endif

                    if(valid)
                    {
                        if(test_inner(Y,parent,degree,rotsys,D,v1+1,previous_mapping)) return 1;  // embed remaining tree
                    }

                    D.mapping_V_to_X[v1] = -1;
                    D.mapping_X_to_V[x1] = -1;
                    for(int y=y0+dy;y!=y1;y+=dy) D.pixel[x0][y] = '.';
                    D.pixel[x0][y1] = '.';
                    for(int x=x0+dx;x!=x1;x+=dx) D.pixel[x][y1] = '.';
                    D.pixel[x1][y1] = '.';

                    #if ORDERED != 0
                        D.order[v0][(dy<0?1:3)] = -1;
                        D.order[v1][(dx<0?0:2)] = -1;
                    #endif
                }
            }
        }
        return 0;
    }
}


// Test whether tree given by parent, degree and rotsys can be embedded in given point set Y.
// Try reusing previous mapping from vertices to x-coordinates.
int test(int Y[N], int parent[N], int degree[N], int rotsys[][4], int previous_mapping[N])
{
    drawing D;
    // initialize drawing
    for(int x=0;x<N;x++)
    {
        for(int y=0;y<N;y++)
        {
            D.pixel[x][y] = '.';
        }
    }
    for(int i=0;i<N;i++)
    {
        D.mapping_V_to_X[i] = -1;
        D.mapping_X_to_V[i] = -1;
        
        #if ORDERED != 0
            for(int j=0;j<4;j++) D.order[i][j] = -1;
        #endif
    }

    // try all possible x-coordinates for vertex 0
    int xii=0;
    for(int xi=0;xi<N;xi++)
    {
        int x = XPERM[xi];
        if(previous_mapping[0] != -1)
        {
            if(xi == 0)
            {
                x = previous_mapping[0];  // reuse x-coordinate from previous embedding
            }
            else 
            {
                if(XPERM[xii] == previous_mapping[0]) xii++;
                x = XPERM[xii];
                xii++;
            }
        }
        
        int y = Y[x];
        if((degree[0] == 4) && (x == 0 || x == N-1 || y == 0 || y == N-1)) continue;  // degree-4 vertices must not be embedded on the boundary

        D.mapping_V_to_X[0] = x;
        D.mapping_X_to_V[x] = 0;
        D.pixel[x][y] = '0';

        if(test_inner(Y,parent,degree,rotsys,D,1,previous_mapping)) return 1; // return if remaining tree can be embedded successfully

        D.mapping_V_to_X[0] = -1;
        D.mapping_X_to_V[x] = -1;
        D.pixel[x][y] = '.';
    }
    return 0;
}


inline void invert_permutation(int* Y,int* Yinv)
{
    for(int i=0;i<N;i++)
    {
        for(int j=0;j<N;j++)
        {
            if(Y[i] == j) Yinv[j] = i;
        }
    }
    for(int i=0;i<N;i++)
    {
        assert(Y[Yinv[i]] == i);
        assert(Yinv[Y[i]] == i);
    }
}


// counterclockwise rotation of point set
inline void rotate_point_set(int* Y,int* Yrot)
{
    for(int i=0;i<N;i++)
    {
        Yrot[N-1-Y[i]] = i;
    }
}


inline int compare_point_sets(int* Y0,int* Y1)
{
    for(int i=0;i<N;i++)
    {
        if(Y0[i] < Y1[i]) return -1;
        if(Y0[i] > Y1[i]) return +1;
    }
    return 0;
}


// Test if given point set, represented as permutation Y0
// that stores the y-coordinates (=heights) of all points, is the
// lexicographically smallest one among all rotations+reflections
// (if ORDERED == 0), or only among all rotations (if ORDERED ==1).
// The former is used for embedding unordered trees, the
// latter is used for embedding ordered trees.
// The number of non-isomorphic point sets is given by
// https://oeis.org/A000903 and https://oeis.org/A263685
// for the two cases, respectively.
inline int is_lexmin_point_set(int Y0[N])
{
    int Y90[N];
    int Y180[N];
    int Y270[N];
    int Y360[N];
    rotate_point_set(Y0,Y90);
    rotate_point_set(Y90,Y180);
    rotate_point_set(Y180,Y270);
    rotate_point_set(Y270,Y360);
    for(int i=0;i<N;i++) assert(Y0[i] == Y360[i]);
    
    if(compare_point_sets(Y0,Y90) > 0) return 0;
    if(compare_point_sets(Y0,Y180) > 0) return 0;
    if(compare_point_sets(Y0,Y270) > 0) return 0; 
    
    #if ORDERED == 0
        // mirror reflection must be tested if cyclic ordering of neighbors is fixed!
        int Y0mir[N];
        int Y90mir[N];
        int Y180mir[N];
        int Y270mir[N];
        // inverting the permutation corresponds to mirroring along the diagonal line y=x
        invert_permutation(Y0,Y0mir);
        invert_permutation(Y90,Y90mir);
        invert_permutation(Y180,Y180mir);
        invert_permutation(Y270,Y270mir);
        if(compare_point_sets(Y0,Y0mir) > 0) return 0;
        if(compare_point_sets(Y0,Y90mir) > 0) return 0;
        if(compare_point_sets(Y0,Y180mir) > 0) return 0;
        if(compare_point_sets(Y0,Y270mir) > 0) return 0; 
    #endif

    return 1;
}


int main_inner(int part,int parts)
{
    std::cout << "[part" << part << "] " << "starting\n";
    
    int Y[N]; // y-coordinates (=heights) of the points from left to right
    int parent[N]; // parent in tree
    int degree[N]; // degree in tree
    int rotsys[N][4]; // rotation system

    // reuse embedding of the same tree on the previous point as a heuristic for the next point set
    int previous_mapping[N];
    previous_mapping[0] = -1;

    #if ORDERED == 0
        std::ifstream infile("unordered_trees/n"+std::to_string(N)+".txt");  
    #else
        std::ifstream infile("nontrivial_ordered_trees/n"+std::to_string(N)+".txt");
    #endif
    std::string line;
    
    std::ofstream outfile;
    
    // loop over all trees in the file
    while (std::getline(infile,line))
    {
        // load next tree
        std::istringstream iss(line);

        for(int i=1;i<N;i++) assert(iss >> parent[i]);

        for(int i=0;i<N;i++) 
        {
            degree[i] = 0;
            if(i > 0) 
            {
                int p = parent[i];
                rotsys[i][degree[i]] = p;
                rotsys[p][degree[p]] = i;
                degree[i]++;
                degree[p]++;
            }
        }

        if(DEBUG > 0)
        {
            print_time();
            std::cout << "parent: ";
            for(int i=1;i<N;i++)  std::cout << parent[i] << " ";
            std::cout << "\n";
        }

        // start with point set on the diagonal line y=x (=identity permutation)
        for(int i=0;i<N;i++) Y[i] = i;

        long ct = 0;

        // loop over all point sets that this thread should consider
        do 
        {
            ct++;
            if(ct % parts != part) continue;  // skip point sets considered by other threads

            if(!is_lexmin_point_set(Y)) continue;  // skip symmetric point sets

            if(DEBUG > 1)
            {
                std::cout << "\tY: ";
                for(int i=0;i<N;i++) std::cout << Y[i];
                std::cout << '\n';
            }

            if(!test(Y,parent,degree,rotsys,previous_mapping))
            {
                std::cout << "ATTENTION: no embedding found!\n";
            
                if(!outfile.is_open())
                {
                    outfile.open ("example"+std::to_string(N)+"_"+std::to_string(part)+"_"+std::to_string(parts)+".txt");
                }
                outfile << "parent: ";
                for(int i=1;i<N;i++) outfile << parent[i] << " ";
                outfile << "\n";
                outfile << "Y: ";
                for(int i=0;i<N;i++) outfile << Y[i] << " ";
                outfile << "\n";
            }
        }
        while(std::next_permutation(Y,Y+N));
    }

    outfile.close();    
    std::cout << "[part" << part << "] " << "complete\n";
    return 0;    
}


int main(int argc,char** argv)
{
    // Initialize permutation XPERM, which contains a priority list of x-coordinates
    // that are used for embedding vertex 0 of the tree. We first try x-coordinates
    // close to the middle, and then closer to the boundaries.
    int m = N/2;
    for(int i=0;i<N;i++) XPERM[i] = m+(1+i)/2 * (i % 2 ? -1 : +1);

    if(DEBUG > 0)
    {
        std::cout << "XPERM: ";
        for(int i=0;i<N;i++) std::cout << XPERM[i];
        std::cout << "\n";
    }

    assert(argc == 6);
    assert(atoi(argv[1]) == ORDERED);
    assert(atoi(argv[2]) == N);
    int parts = atoi(argv[3]); 
    int part0 = atoi(argv[4]);
    int part1 = atoi(argv[5]);
    assert(0 <= part0 && part0 < part1 && part1 <= parts);

    int num_threads = part1-part0;
    std::thread* threads = new std::thread[num_threads];
    std::cout << "starting threads: " << num_threads << "\n";

    for(int i = 0; i<num_threads; i++) 
    {
        threads[i] = std::thread(main_inner,part0+i,parts);
    }

    for(int i = 0; i<num_threads; i++) 
    {
        threads[i].join();
    }
    delete[] threads;

    std::cout << "all threads done.\n";
}
